<?php
require_once '../config/db.php';

// Format file size function
function formatBoyut($bytes) {
    if ($bytes >= 1073741824) {
        return number_format($bytes / 1073741824, 2) . ' GB';
    } elseif ($bytes >= 1048576) {
        return number_format($bytes / 1048576, 2) . ' MB';
    } elseif ($bytes >= 1024) {
        return number_format($bytes / 1024, 2) . ' KB';
    }
    return $bytes . ' B';
}

// Get backup settings
$sorgu = $db->query("SELECT * FROM yedekleme_ayarlar WHERE otomatik_yedekleme = 1 LIMIT 1");
$ayarlar = $sorgu->fetch(PDO::FETCH_ASSOC);

if (!$ayarlar) {
    die("Otomatik yedekleme devre dışı.");
}

// Check if it's time to backup
$now = new DateTime();
$last_backup = $ayarlar['son_yedekleme'] ? new DateTime($ayarlar['son_yedekleme']) : null;
$backup_time = new DateTime($ayarlar['yedekleme_zamani']);

$should_backup = false;

switch ($ayarlar['yedekleme_siklik']) {
    case 'daily':
        $should_backup = !$last_backup || $last_backup->format('Y-m-d') != $now->format('Y-m-d');
        break;
    case 'weekly':
        $should_backup = !$last_backup || $last_backup->format('Y-W') != $now->format('Y-W');
        break;
    case 'monthly':
        $should_backup = !$last_backup || $last_backup->format('Y-m') != $now->format('Y-m');
        break;
}

if (!$should_backup) {
    die("Yedekleme zamanı değil.");
}

// Backup directory
$backup_dir = __DIR__ . '/../backups';
if (!file_exists($backup_dir)) {
    mkdir($backup_dir, 0755, true);
}

// Start backup process
$baslangic_zamani = date('Y-m-d H:i:s');
$hata_mesaji = null;
$dosya_adi = null;
$boyut = 0;

try {
    $timestamp = date('Y-m-d_H-i-s');
    $backup_path = $backup_dir . '/backup_' . $ayarlar['yedekleme_turu'] . '_' . $timestamp;
    
    // Create backup directory
    if (!file_exists($backup_path)) {
        mkdir($backup_path, 0755, true);
    }
    
    // Database backup
    if ($ayarlar['yedekleme_turu'] == 'full' || $ayarlar['yedekleme_turu'] == 'database') {
        $tables = [];
        $result = $db->query("SHOW TABLES");
        while ($row = $result->fetch(PDO::FETCH_NUM)) {
            $tables[] = $row[0];
        }
        
        $sql_file = $backup_path . '/database.sql';
        $handle = fopen($sql_file, 'w');
        
        // Add DROP TABLE statements
        foreach ($tables as $table) {
            fwrite($handle, "DROP TABLE IF EXISTS `$table`;\n");
        }
        
        // Get table structures and data
        foreach ($tables as $table) {
            // Get create table statement
            $result = $db->query("SHOW CREATE TABLE `$table`");
            $row = $result->fetch(PDO::FETCH_NUM);
            fwrite($handle, "\n" . $row[1] . ";\n\n");
            
            // Get table data
            $result = $db->query("SELECT * FROM `$table`");
            while ($row = $result->fetch(PDO::FETCH_NUM)) {
                $values = array_map(function($value) use ($db) {
                    if ($value === null) return 'NULL';
                    return "'" . $db->quote($value) . "'";
                }, $row);
                
                fwrite($handle, "INSERT INTO `$table` VALUES (" . implode(',', $values) . ");\n");
            }
            fwrite($handle, "\n");
        }
        
        fclose($handle);
    }
    
    // Files backup
    if ($ayarlar['yedekleme_turu'] == 'full' || $ayarlar['yedekleme_turu'] == 'files') {
        $source = __DIR__ . '/../';
        $dest = $backup_path . '/files';
        
        // Create recursive directory iterator
        $files = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator($source),
            RecursiveIteratorIterator::LEAVES_ONLY
        );
        
        foreach ($files as $file) {
            // Skip directories and backup folder
            if ($file->isDir() || strpos($file->getPathname(), 'backups') !== false) {
                continue;
            }
            
            $file_path = $file->getRealPath();
            $relative_path = substr($file_path, strlen($source));
            $dest_path = $dest . '/' . $relative_path;
            
            // Create directory if it doesn't exist
            $dest_dir = dirname($dest_path);
            if (!file_exists($dest_dir)) {
                mkdir($dest_dir, 0755, true);
            }
            
            copy($file_path, $dest_path);
        }
    }
    
    // Create zip archive
    $zip = new ZipArchive();
    $zip_file = $backup_dir . '/backup_' . $ayarlar['yedekleme_turu'] . '_' . $timestamp . '.zip';
    
    if ($zip->open($zip_file, ZipArchive::CREATE | ZipArchive::OVERWRITE) === true) {
        $files = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator($backup_path),
            RecursiveIteratorIterator::LEAVES_ONLY
        );
        
        foreach ($files as $file) {
            if (!$file->isDir()) {
                $file_path = $file->getRealPath();
                $relative_path = substr($file_path, strlen($backup_path) + 1);
                $zip->addFile($file_path, $relative_path);
            }
        }
        
        $zip->close();
        
        // Remove temporary backup directory
        rrmdir($backup_path);
        
        $dosya_adi = basename($zip_file);
        $boyut = filesize($zip_file);
        
        // Clean old backups
        $files = scandir($backup_dir);
        foreach ($files as $file) {
            if ($file != '.' && $file != '..' && pathinfo($file, PATHINFO_EXTENSION) == 'zip') {
                $file_path = $backup_dir . '/' . $file;
                if (time() - filemtime($file_path) > $ayarlar['sakla_gun'] * 86400) {
                    unlink($file_path);
                }
            }
        }
        
        // Update last backup time
        $db->prepare("UPDATE yedekleme_ayarlar SET son_yedekleme = NOW() WHERE id = ?")->execute([$ayarlar['id']]);
    } else {
        throw new Exception('Zip dosyası oluşturulamadı.');
    }
    
} catch (Exception $e) {
    $hata_mesaji = $e->getMessage();
}

// Log backup result
$bitis_zamani = date('Y-m-d H:i:s');
$durum = $hata_mesaji ? 'failed' : 'success';

$sorgu = $db->prepare("INSERT INTO yedekleme_log 
    (yedekleme_turu, dosya_adi, boyut, durum, hata_mesaji, baslangic_zamani, bitis_zamani) 
    VALUES (?, ?, ?, ?, ?, ?, ?)");

$sorgu->execute([
    $ayarlar['yedekleme_turu'],
    $dosya_adi,
    $boyut,
    $durum,
    $hata_mesaji,
    $baslangic_zamani,
    $bitis_zamani
]);

// Recursive directory removal
function rrmdir($dir) {
    if (is_dir($dir)) {
        $objects = scandir($dir);
        foreach ($objects as $object) {
            if ($object != "." && $object != "..") {
                if (is_dir($dir . "/" . $object)) {
                    rrmdir($dir . "/" . $object);
                } else {
                    unlink($dir . "/" . $object);
                }
            }
        }
        rmdir($dir);
    }
}

if ($hata_mesaji) {
    die("Yedekleme hatası: " . $hata_mesaji);
} else {
    echo "Yedekleme başarıyla tamamlandı: " . $dosya_adi;
}
?>