<?php
ob_start();
oturumKontrol();

// Format file size function
function formatBoyut($bytes) {
    if ($bytes >= 1073741824) {
        return number_format($bytes / 1073741824, 2) . ' GB';
    } elseif ($bytes >= 1048576) {
        return number_format($bytes / 1048576, 2) . ' MB';
    } elseif ($bytes >= 1024) {
        return number_format($bytes / 1024, 2) . ' KB';
    }
    return $bytes . ' B';
}

// Create backup settings table if it doesn't exist
$db->query("CREATE TABLE IF NOT EXISTS yedekleme_ayarlar (
    id INT PRIMARY KEY AUTO_INCREMENT,
    otomatik_yedekleme BOOLEAN NOT NULL DEFAULT false,
    yedekleme_siklik ENUM('daily', 'weekly', 'monthly') NOT NULL DEFAULT 'daily',
    yedekleme_zamani TIME NOT NULL DEFAULT '00:00:00',
    yedekleme_turu ENUM('full', 'database', 'files') NOT NULL DEFAULT 'full',
    sakla_gun INT NOT NULL DEFAULT 30,
    son_yedekleme DATETIME NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
)");

// Sabit yol tanımlamaları
define('BACKUP_DIR', ROOT_PATH . '/backups');
define('SITE_FILES_DIR', ROOT_PATH);

// Veritabanı yedekleme fonksiyonu
function backupDatabase($backup_path) {
    global $db;
    
    try {
        // Veritabanı bağlantı bilgilerini al
        $dbname = DB_NAME;
        $dbuser = DB_USER;
        $dbpass = DB_PASS;
        $dbhost = DB_HOST;
        
        // Yedek dosyası oluştur
        $backup_file = $backup_path . '/database.sql';
        
        // mysqldump komutu oluştur
        $command = sprintf(
            'mysqldump --opt -h %s -u %s -p%s %s > %s',
            escapeshellarg($dbhost),
            escapeshellarg($dbuser),
            escapeshellarg($dbpass),
            escapeshellarg($dbname),
            escapeshellarg($backup_file)
        );
        
        // Komutu çalıştır
        exec($command, $output, $return_var);
        
        if ($return_var !== 0) {
            throw new Exception('Veritabanı yedeği alınamadı.');
        }
        
        return true;
    } catch (Exception $e) {
        // Hata durumunda manuel yedekleme yap
        $sql = "-- Veritabanı Yedeği\n";
        $sql .= "-- Tarih: " . date('Y-m-d H:i:s') . "\n\n";
        
        // Tablo listesini al
        $tables = $db->query("SHOW TABLES")->fetchAll(PDO::FETCH_COLUMN);
        
        foreach ($tables as $table) {
            // Tablo yapısı
            $create = $db->query("SHOW CREATE TABLE `$table`")->fetch(PDO::FETCH_ASSOC);
            $sql .= "\n\n" . $create['Create Table'] . ";\n\n";
            
            // Tablo verileri
            $rows = $db->query("SELECT * FROM `$table`")->fetchAll(PDO::FETCH_ASSOC);
            foreach ($rows as $row) {
                $values = array_map(function($value) {
                    if ($value === null) {
                        return 'NULL';
                    }
                    return "'" . addslashes($value) . "'";
                }, $row);
                
                $sql .= "INSERT INTO `$table` (`" . implode("`, `", array_keys($row)) . "`) VALUES (" . implode(", ", $values) . ");\n";
            }
            $sql .= "\n";
        }
        
        // SQL dosyasını kaydet
        file_put_contents($backup_path . '/database.sql', $sql);
        
        return true;
    }
}

// Klasör kopyalama fonksiyonu
function copyDirectory($source, $dest) {
    if (!is_dir($source)) {
        return false;
    }
    
    if (!is_dir($dest)) {
        mkdir($dest, 0777, true);
    }
    
    $dir = dir($source);
    while (false !== ($entry = $dir->read())) {
        if ($entry == '.' || $entry == '..' || $entry == 'backups') {
            continue;
        }
        
        $src = $source . '/' . $entry;
        $dst = $dest . '/' . $entry;
        
        if (is_dir($src)) {
            copyDirectory($src, $dst);
        } else {
            copy($src, $dst);
        }
    }
    $dir->close();
    return true;
}

// ZIP oluşturma fonksiyonu
function createZipArchive($source, $destination) {
    if (!extension_loaded('zip')) {
        return false;
    }

    $zip = new ZipArchive();
    if (!$zip->open($destination, ZIPARCHIVE::CREATE | ZIPARCHIVE::OVERWRITE)) {
        return false;
    }
    
    $source = str_replace('\\', '/', realpath($source));
    
    if (is_dir($source)) {
        $files = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator($source),
            RecursiveIteratorIterator::LEAVES_ONLY
        );

        foreach ($files as $file) {
            if (!$file->isDir()) {
                $filePath = $file->getRealPath();
                $relativePath = substr(str_replace('\\', '/', $filePath), strlen($source) + 1);

                $zip->addFile($filePath, $relativePath);
            }
        }
    }
    
    $zip->close();
    return true;
}

// Klasör silme fonksiyonu
function deleteDirectory($dir) {
    if (!file_exists($dir)) {
        return true;
    }
    
    if (!is_dir($dir)) {
        return unlink($dir);
    }
    
    foreach (scandir($dir) as $item) {
        if ($item == '.' || $item == '..') {
            continue;
        }
        
        if (!deleteDirectory($dir . DIRECTORY_SEPARATOR . $item)) {
            return false;
        }
    }
    
    return rmdir($dir);
}

// Yedek alma fonksiyonu
function createBackup() {
    $backup_date = date('Y-m-d_H-i-s');
    $backup_name = "backup_full_" . $backup_date;
    $backup_path = BACKUP_DIR . '/' . $backup_name;
    
    // Yedekleme klasörü yoksa oluştur
    if (!file_exists(BACKUP_DIR)) {
        mkdir(BACKUP_DIR, 0777, true);
    }

    // Yedekleme klasörünü oluştur
    mkdir($backup_path, 0777, true);
    mkdir($backup_path . '/files', 0777, true);
    
    // Dosyaları kopyala
    copyDirectory(SITE_FILES_DIR, $backup_path . '/files');
    
    // Veritabanı yedeği al
    backupDatabase($backup_path);
    
    // ZIP oluştur
    createZipArchive($backup_path, BACKUP_DIR . '/' . $backup_name . '.zip');
    
    // Geçici klasörü sil
    deleteDirectory($backup_path);
    
    return [
    'success' => true,
    'file' => $backup_name . '.zip'
    ];
}

// Handle backup creation
if (isset($_POST['create_backup'])) {
    $type = $_POST['backup_type'];
    $result = createBackup($type);
    
    if ($result['success']) {
        $basarili = 'Yedekleme başarıyla oluşturuldu: ' . $result['file'];
    } else {
        $hata = 'Yedekleme oluşturulurken hata: ' . $result['error'];
    }
}

// Handle backup deletion
if (isset($_POST['delete_backup'])) {
    $file = $_POST['file'];
    $file_path = BACKUP_DIR . '/' . basename($file);
    
    if (file_exists($file_path) && unlink($file_path)) {
        $basarili = 'Yedek dosyası silindi.';
    } else {
        $hata = 'Yedek dosyası silinemedi.';
    }
}

// Get backup list
$backups = [];
if (is_dir(BACKUP_DIR)) {
    $files = scandir(BACKUP_DIR);
    foreach ($files as $file) {
        if ($file != '.' && $file != '..' && pathinfo($file, PATHINFO_EXTENSION) == 'zip') {
            $file_path = BACKUP_DIR . '/' . $file;
            $backups[] = [
                'file' => $file,
                'size' => filesize($file_path),
                'date' => date('Y-m-d H:i:s', filemtime($file_path))
            ];
        }
    }
}

// Sort backups by date (newest first)
usort($backups, function($a, $b) {
    return strtotime($b['date']) - strtotime($a['date']);
});

// Get backup schedule settings
$sorgu = $db->query("SELECT * FROM yedekleme_ayarlar LIMIT 1");
$ayarlar = $sorgu->fetch(PDO::FETCH_ASSOC) ?: [
    'otomatik_yedekleme' => false,
    'yedekleme_siklik' => 'daily',
    'yedekleme_zamani' => '00:00:00',
    'yedekleme_turu' => 'full',
    'sakla_gun' => 30
];

// Update backup schedule settings
if (isset($_POST['update_schedule'])) {
    $otomatik_yedekleme = isset($_POST['otomatik_yedekleme']) ? 1 : 0;
    $yedekleme_siklik = $_POST['yedekleme_siklik'];
    $yedekleme_zamani = $_POST['yedekleme_zamani'];
    $yedekleme_turu = $_POST['yedekleme_turu'];
    $sakla_gun = (int)$_POST['sakla_gun'];
    
    try {
        if ($ayarlar['id'] ?? false) {
            $sorgu = $db->prepare("UPDATE yedekleme_ayarlar SET 
                otomatik_yedekleme = ?, 
                yedekleme_siklik = ?, 
                yedekleme_zamani = ?,
                yedekleme_turu = ?,
                sakla_gun = ?
                WHERE id = ?");
            
            $sonuc = $sorgu->execute([
                $otomatik_yedekleme,
                $yedekleme_siklik,
                $yedekleme_zamani,
                $yedekleme_turu,
                $sakla_gun,
                $ayarlar['id']
            ]);
        } else {
            $sorgu = $db->prepare("INSERT INTO yedekleme_ayarlar 
                (otomatik_yedekleme, yedekleme_siklik, yedekleme_zamani, yedekleme_turu, sakla_gun) 
                VALUES (?, ?, ?, ?, ?)");
            
            $sonuc = $sorgu->execute([
                $otomatik_yedekleme,
                $yedekleme_siklik,
                $yedekleme_zamani,
                $yedekleme_turu,
                $sakla_gun
            ]);
        }
        
        if ($sonuc) {
            $basarili = 'Yedekleme ayarları güncellendi.';
            header("Refresh:2");
        }
    } catch (PDOException $e) {
        $hata = 'Ayarlar güncellenirken hata: ' . $e->getMessage();
    }
}
?>

<!-- Breadcrumb -->
<nav aria-label="breadcrumb">
    <ol class="breadcrumb">
        <li class="breadcrumb-item"><a href="index.php">Anasayfa</a></li>
        <li class="breadcrumb-item"><a href="index.php?sayfa=ayarlar">Ayarlar</a></li>
        <li class="breadcrumb-item active">Yedekleme Ayarları</li>
    </ol>
</nav>

<!-- Messages -->
<?php if (isset($hata)): ?>
    <div class="alert alert-danger">
        <i class="bi bi-exclamation-triangle-fill me-2"></i>
        <?php echo $hata; ?>
    </div>
<?php endif; ?>

<?php if (isset($basarili)): ?>
    <div class="alert alert-success">
        <i class="bi bi-check-circle-fill me-2"></i>
        <?php echo $basarili; ?>
    </div>
<?php endif; ?>

<div class="row">
    <!-- Backup Creation -->
    <div class="col-md-6 mb-4">
        <div class="card h-100">
            <div class="card-header">
                <h5 class="card-title mb-0">
                    <i class="bi bi-cloud-upload me-2"></i>
                    Manuel Yedekleme
                </h5>
            </div>
            <div class="card-body">
                <form method="POST" action="" class="mb-4">
                    <input type="hidden" name="create_backup" value="1">
                    
                    <div class="mb-3">
                        <label for="backup_type" class="form-label">Yedekleme Türü</label>
                        <select class="form-select" id="backup_type" name="backup_type" required>
                            <option value="full">Tam Yedek (Veritabanı + Dosyalar)</option>
                            <option value="database">Sadece Veritabanı</option>
                            <option value="files">Sadece Dosyalar</option>
                        </select>
                    </div>
                    
                    <button type="submit" class="btn btn-primary">
                        <i class="bi bi-cloud-upload me-2"></i>
                        Yedekleme Oluştur
                    </button>
                </form>
                
                <div class="alert alert-info">
                    <h6 class="alert-heading">
                        <i class="bi bi-info-circle me-2"></i>
                        Yedekleme Hakkında
                    </h6>
                    <p class="mb-0">
                        Tam yedekleme tüm veritabanını ve dosyaları içerir. Yedekleme işlemi sistem yüküne ve veri boyutuna bağlı olarak birkaç dakika sürebilir.
                    </p>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Backup Schedule -->
    <div class="col-md-6 mb-4">
        <div class="card h-100">
            <div class="card-header">
                <h5 class="card-title mb-0">
                    <i class="bi bi-clock-history me-2"></i>
                    Otomatik Yedekleme
                </h5>
            </div>
            <div class="card-body">
                <form method="POST" action="">
                    <input type="hidden" name="update_schedule" value="1">
                    
                    <div class="form-check form-switch mb-3">
                        <input class="form-check-input" type="checkbox" id="otomatik_yedekleme" name="otomatik_yedekleme"
                               <?php echo $ayarlar['otomatik_yedekleme'] ? 'checked' : ''; ?>>
                        <label class="form-check-label" for="otomatik_yedekleme">
                            Otomatik Yedeklemeyi Etkinleştir
                        </label>
                    </div>
                    
                    <div class="mb-3">
                        <label for="yedekleme_siklik" class="form-label">Yedekleme Sıklığı</label>
                        <select class="form-select" id="yedekleme_siklik" name="yedekleme_siklik" required>
                            <option value="daily" <?php echo $ayarlar['yedekleme_siklik'] == 'daily' ? 'selected' : ''; ?>>Her Gün</option>
                            <option value="weekly" <?php echo $ayarlar['yedekleme_siklik'] == 'weekly' ? 'selected' : ''; ?>>Haftalık</option>
                            <option value="monthly" <?php echo $ayarlar['yedekleme_siklik'] == 'monthly' ? 'selected' : ''; ?>>Aylık</option>
                        </select>
                    </div>
                    
                    <div class="mb-3">
                        <label for="yedekleme_zamani" class="form-label">Yedekleme Zamanı</label>
                        <input type="time" class="form-control" id="yedekleme_zamani" name="yedekleme_zamani" 
                               value="<?php echo $ayarlar['yedekleme_zamani']; ?>" required>
                    </div>
                    
                    <div class="mb-3">
                        <label for="yedekleme_turu" class="form-label">Yedekleme Türü</label>
                        <select class="form-select" id="yedekleme_turu" name="yedekleme_turu" required>
                            <option value="full" <?php echo $ayarlar['yedekleme_turu'] == 'full' ? 'selected' : ''; ?>>Tam Yedek</option>
                            <option value="database" <?php echo $ayarlar['yedekleme_turu'] == 'database' ? 'selected' : ''; ?>>Sadece Veritabanı</option>
                            <option value="files" <?php echo $ayarlar['yedekleme_turu'] == 'files' ? 'selected' : ''; ?>>Sadece Dosyalar</option>
                        </select>
                    </div>
                    
                    <div class="mb-3">
                        <label for="sakla_gun" class="form-label">Yedekleri Saklama Süresi (Gün)</label>
                        <input type="number" class="form-control" id="sakla_gun" name="sakla_gun" 
                               value="<?php echo $ayarlar['sakla_gun']; ?>" min="1" required>
                        <small class="text-muted">Bu süreden eski yedekler otomatik silinir.</small>
                    </div>
                    
                    <button type="submit" class="btn btn-primary">
                        <i class="bi bi-save me-2"></i>
                        Ayarları Kaydet
                    </button>
                </form>
            </div>
        </div>
    </div>
</div>

<!-- Backup List -->
<div class="card">
    <div class="card-header">
        <h5 class="card-title mb-0">
            <i class="bi bi-archive me-2"></i>
            Yedek Listesi
        </h5>
    </div>
    <div class="card-body">
        <?php if (empty($backups)): ?>
            <div class="alert alert-info mb-0">
                <i class="bi bi-info-circle me-2"></i>
                Henüz yedek bulunmamaktadır.
            </div>
        <?php else: ?>
            <div class="table-responsive">
                <table class="table table-hover">
                    <thead>
                        <tr>
                            <th>Dosya Adı</th>
                            <th>Boyut</th>
                            <th>Tarih</th>
                            <th>İşlemler</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($backups as $backup): ?>
                        <tr>
                            <td>
                                <i class="bi bi-file-earmark-zip me-2"></i>
                                <?php echo htmlspecialchars($backup['file']); ?>
                            </td>
                            <td><?php echo formatBoyut($backup['size']); ?></td>
                            <td><?php echo date('d.m.Y H:i', strtotime($backup['date'])); ?></td>
                            <td>
                                <div class="btn-group">
                                    <a href="backups/<?php echo urlencode($backup['file']); ?>" 
                                       class="btn btn-primary btn-sm" download>
                                        <i class="bi bi-download me-1"></i>
                                        İndir
                                    </a>
                                    <button type="button" class="btn btn-danger btn-sm" 
                                            onclick="yedekSil('<?php echo htmlspecialchars($backup['file']); ?>')">
                                        <i class="bi bi-trash me-1"></i>
                                        Sil
                                    </button>
                                </div>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        <?php endif; ?>
    </div>
</div>

<!-- Delete Form -->
<form id="silForm" method="POST" style="display: none;">
    <input type="hidden" name="delete_backup" value="1">
    <input type="hidden" name="file" id="sil_file">
</form>

<script>
// Yedek silme işlemi
function yedekSil(file) {
    if (confirm('Bu yedek dosyasını silmek istediğinizden emin misiniz?')) {
        document.getElementById('sil_file').value = file;
        document.getElementById('silForm').submit();
    }
}

// Otomatik yedekleme ayarlarını kontrol et
document.addEventListener('DOMContentLoaded', function() {
    const otomatikYedekleme = document.getElementById('otomatik_yedekleme');
    const yedeklemeAyarlari = document.querySelectorAll('#yedekleme_siklik, #yedekleme_zamani, #yedekleme_turu, #sakla_gun');
    
    function ayarlariGuncelle() {
        yedeklemeAyarlari.forEach(ayar => {
            ayar.disabled = !otomatikYedekleme.checked;
        });
    }
    
    otomatikYedekleme.addEventListener('change', ayarlariGuncelle);
    ayarlariGuncelle();
});
</script>

<?php ob_end_flush(); ?>