<?php
require_once '../config/db.php';

// Get notification settings
$sorgu = $db->query("SELECT * FROM bildirim_ayarlar LIMIT 1");
$bildirim_ayarlar = $sorgu->fetch(PDO::FETCH_ASSOC);

// Get invoice notification template
$sorgu = $db->query("SELECT * FROM bildirim_sablon WHERE tur = 'fatura' AND otomatik = 1 LIMIT 1");
$fatura_sablon = $sorgu->fetch(PDO::FETCH_ASSOC);

// UUID oluşturma fonksiyonu
function generateUUID() {
    return sprintf('%04x%04x-%04x-%04x-%04x-%04x%04x%04x',
        mt_rand(0, 0xffff), mt_rand(0, 0xffff),
        mt_rand(0, 0xffff),
        mt_rand(0, 0x0fff) | 0x4000,
        mt_rand(0, 0x3fff) | 0x8000,
        mt_rand(0, 0xffff), mt_rand(0, 0xffff), mt_rand(0, 0xffff)
    );
}

// Log fonksiyonu
function logYaz($mesaj) {
    $log_dosyasi = __DIR__ . '/fatura_log.txt';
    $tarih = date('Y-m-d H:i:s');
    file_put_contents($log_dosyasi, "[$tarih] $mesaj\n", FILE_APPEND);
}

try {
    $db->beginTransaction();
    
    // Vadesi gelen hosting hesaplarını bul
    $hosting_sorgu = $db->prepare("
        SELECT h.*, m.ad, m.soyad, m.eposta, hp.fiyat as paket_fiyat
        FROM hosting_hesaplari h
        LEFT JOIN musteriler m ON h.musteri_id = m.id
        LEFT JOIN hosting_paketleri hp ON h.paket_id = hp.id
        WHERE h.durum = 'Aktif' 
        AND h.sonraki_odeme <= CURDATE()
        AND h.fatura_donemi != 'Ücretsiz'
        AND NOT EXISTS (
            SELECT 1 FROM faturalar f 
            JOIN fatura_kalemleri fk ON f.id = fk.fatura_id 
            WHERE fk.hizmet_id = h.id 
            AND fk.hizmet_turu = 'Hosting'
            AND f.fatura_tarihi = CURDATE()
        )
    ");
    $hosting_sorgu->execute();
    $hosting_hesaplari = $hosting_sorgu->fetchAll(PDO::FETCH_ASSOC);
    
    // Vadesi gelen domainleri bul
    $domain_sorgu = $db->prepare("
        SELECT d.*, m.ad, m.soyad, m.eposta, df.yenileme_fiyat
        FROM domain_kayitlar d
        LEFT JOIN musteriler m ON d.musteri_id = m.id
        LEFT JOIN domain_fiyatlar df ON df.uzanti = SUBSTRING_INDEX(d.domain, '.', -2)
        WHERE d.durum = 'Aktif' 
        AND d.bitis_tarihi <= DATE_ADD(CURDATE(), INTERVAL 30 DAY)
        AND NOT EXISTS (
            SELECT 1 FROM faturalar f 
            JOIN fatura_kalemleri fk ON f.id = fk.fatura_id 
            WHERE fk.hizmet_id = d.id 
            AND fk.hizmet_turu = 'Domain'
            AND f.fatura_tarihi = CURDATE()
        )
    ");
    $domain_sorgu->execute();
    $domainler = $domain_sorgu->fetchAll(PDO::FETCH_ASSOC);
    
    // Müşteri bazında fatura oluştur
    $musteriler = [];
    
    // Hosting hesaplarını grupla
    foreach ($hosting_hesaplari as $hosting) {
        if (!isset($musteriler[$hosting['musteri_id']])) {
            $musteriler[$hosting['musteri_id']] = [
                'ad' => $hosting['ad'],
                'soyad' => $hosting['soyad'],
                'eposta' => $hosting['eposta'],
                'hosting' => [],
                'domain' => []
            ];
        }
        $musteriler[$hosting['musteri_id']]['hosting'][] = $hosting;
    }
    
    // Domainleri grupla
    foreach ($domainler as $domain) {
        if (!isset($musteriler[$domain['musteri_id']])) {
            $musteriler[$domain['musteri_id']] = [
                'ad' => $domain['ad'],
                'soyad' => $domain['soyad'],
                'eposta' => $domain['eposta'],
                'hosting' => [],
                'domain' => []
            ];
        }
        $musteriler[$domain['musteri_id']]['domain'][] = $domain;
    }
    
    // Her müşteri için fatura oluştur
    foreach ($musteriler as $musteri_id => $musteri) {
        if (empty($musteri['hosting']) && empty($musteri['domain'])) {
            continue;
        }
        
        // Fatura numarası oluştur
        $yil = date('Y');
        $sayac_sorgu = $db->prepare("INSERT INTO fatura_sayac (yil, son_numara) 
                                    VALUES (?, 0) 
                                    ON DUPLICATE KEY UPDATE son_numara = son_numara + 1");
        $sayac_sorgu->execute([$yil]);
        
        $sorgu = $db->prepare("SELECT son_numara FROM fatura_sayac WHERE yil = ?");
        $sorgu->execute([$yil]);
        $son_numara = $sorgu->fetchColumn();
        
        $fatura_no = sprintf("%s-%04d", $yil, $son_numara);
        
        // Fatura başlığını oluştur
        $fatura_id = generateUUID();
        $fatura_tarihi = date('Y-m-d');
        $son_odeme_tarihi = date('Y-m-d', strtotime('+7 days'));
        
        $sorgu = $db->prepare("INSERT INTO faturalar 
            (id, musteri_id, fatura_no, fatura_tarihi, son_odeme_tarihi) 
            VALUES (?, ?, ?, ?, ?)");
        
        $sorgu->execute([
            $fatura_id,
            $musteri_id,
            $fatura_no,
            $fatura_tarihi,
            $son_odeme_tarihi
        ]);
        
        $ara_toplam = 0;
        
        // Hosting kalemlerini ekle
        foreach ($musteri['hosting'] as $hosting) {
            $kalem_id = generateUUID();
            
            $sorgu = $db->prepare("INSERT INTO fatura_kalemleri 
                (id, fatura_id, hizmet_id, hizmet_turu, aciklama, birim, miktar, birim_fiyat, toplam) 
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)");
            
            // Paket fiyatını veya varsayılan fiyatı kullan
            $fiyat = $hosting['paket_fiyat'] ?: $hosting['fiyat'];
            $toplam = $fiyat;
            
            $sorgu->execute([
                $kalem_id,
                $fatura_id,
                $hosting['id'],
                'Hosting',
                $hosting['domain'] . ' Hosting Yenileme (' . $hosting['paket'] . ')',
                'Yıl',
                1,
                $fiyat,
                $toplam
            ]);
            
            $ara_toplam += $toplam;
            
            // Sonraki ödeme tarihini güncelle
            $yeni_tarih = date('Y-m-d', strtotime('+1 year', strtotime($hosting['sonraki_odeme'])));
            $sorgu = $db->prepare("UPDATE hosting_hesaplari SET sonraki_odeme = ? WHERE id = ?");
            $sorgu->execute([$yeni_tarih, $hosting['id']]);
        }
        
        // Domain kalemlerini ekle
        foreach ($musteri['domain'] as $domain) {
            $kalem_id = generateUUID();
            
            $sorgu = $db->prepare("INSERT INTO fatura_kalemleri 
                (id, fatura_id, hizmet_id, hizmet_turu, aciklama, birim, miktar, birim_fiyat, toplam) 
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)");
            
            // Domain uzantısına göre fiyatı al veya varsayılan fiyatı kullan
            $fiyat = $domain['yenileme_fiyat'] ?: 150.00;
            $toplam = $fiyat;
            
            $sorgu->execute([
                $kalem_id,
                $fatura_id,
                $domain['id'],
                'Domain',
                $domain['domain'] . ' Domain Yenileme',
                'Yıl',
                1,
                $fiyat,
                $toplam
            ]);
            
            $ara_toplam += $toplam;
        }
        
        // KDV hesapla ve faturayı güncelle
        $kdv_oran = 20.00;
        $kdv_tutar = $ara_toplam * ($kdv_oran / 100);
        $genel_toplam = $ara_toplam + $kdv_tutar;
        
        $sorgu = $db->prepare("UPDATE faturalar SET 
            ara_toplam = ?, 
            kdv_oran = ?, 
            kdv_tutar = ?, 
            genel_toplam = ? 
            WHERE id = ?");
        
        $sorgu->execute([
            $ara_toplam,
            $kdv_oran,
            $kdv_tutar,
            $genel_toplam,
            $fatura_id
        ]);
        
        // Online ödeme kaydı oluştur
        $online_odeme_id = generateUUID();
        $sorgu = $db->prepare("INSERT INTO online_odemeler (id, fatura_id, tutar) VALUES (?, ?, ?)");
        $sorgu->execute([$online_odeme_id, $fatura_id, $genel_toplam]);

        if ($bildirim_ayarlar && $fatura_sablon) {
            // Replace template variables
            $email_content = $fatura_sablon['icerik'];
            $sms_content = $fatura_sablon['sms_icerik'];
            
            $variables = [
                '{AD}' => $musteri['ad'],
                '{SOYAD}' => $musteri['soyad'],
                '{FATURA_NO}' => $fatura_no,
                '{TUTAR}' => number_format($genel_toplam, 2, ',', '.'),
                '{FATURA_TARIHI}' => date('d.m.Y', strtotime($fatura_tarihi)),
                '{SON_ODEME_TARIHI}' => date('d.m.Y', strtotime($son_odeme_tarihi)),
                '{ODEME_LINK}' => SITE_URL . "/odeme.php?id=" . $online_odeme_id
            ];
            
            foreach ($variables as $key => $value) {
                $email_content = str_replace($key, $value, $email_content);
                $sms_content = str_replace($key, $value, $sms_content);
            }
            
            // Send email
            require_once 'inc/phpmailer/PHPMailer.php';
            require_once 'inc/phpmailer/SMTP.php';
            require_once 'inc/phpmailer/Exception.php';
            
            try {
                $mail = new PHPMailer\PHPMailer\PHPMailer(true);
                
                // Server settings
                $mail->isSMTP();
                $mail->Host = $bildirim_ayarlar['smtp_host'];
                $mail->SMTPAuth = true;
                $mail->Username = $bildirim_ayarlar['smtp_user'];
                $mail->Password = $bildirim_ayarlar['smtp_pass'];
                $mail->SMTPSecure = PHPMailer\PHPMailer\PHPMailer::ENCRYPTION_STARTTLS;
                $mail->Port = $bildirim_ayarlar['smtp_port'];
                $mail->CharSet = 'UTF-8';
                
                // Sender and recipient
                $mail->setFrom($bildirim_ayarlar['smtp_from'], $bildirim_ayarlar['smtp_from_name']);
                $mail->addAddress($musteri['eposta']);
                
                // Content
                $mail->isHTML(true);
                $mail->Subject = $fatura_sablon['baslik'];
                $mail->Body = $email_content;
                
                // Add PDF attachment
                $mail->addStringAttachment($pdf_content, 'fatura.pdf');
                
                // Log before sending
                $log_data = [
                    'tur' => 'email',
                    'alici' => $musteri['eposta'],
                    'baslik' => $fatura_sablon['baslik'],
                    'icerik' => $email_content,
                    'durum' => 'pending'
                ];
                
                $log_id = logBildirim($log_data);
                
                // Send email
                $success = $mail->send();
                
                // Update log
                updateBildirimLog($log_id, $success ? 'success' : 'failed', $success ? null : $mail->ErrorInfo);
                
                // Send SMS if enabled
                if ($success && $bildirim_ayarlar['sms_user'] && $bildirim_ayarlar['sms_pass']) {
                    sendSMS($musteri['telefon'], $sms_content, $bildirim_ayarlar);
                }
                
            } catch (Exception $e) {
                error_log("Fatura e-posta gönderme hatası: " . $e->getMessage());
                if (isset($log_id)) {
                    updateBildirimLog($log_id, 'failed', $e->getMessage());
                }
            }
        }
    }
    
    $db->commit();
    logYaz("İşlem başarıyla tamamlandı.");
    
} catch (Exception $e) {
    $db->rollBack();
    logYaz("HATA: " . $e->getMessage());
}

// Helper functions for notification logging
function logBildirim($data) {
    global $db;
    
    $sorgu = $db->prepare("INSERT INTO bildirim_log 
        (tur, alici, baslik, icerik, durum) 
        VALUES (?, ?, ?, ?, 'failed')");
    
    $sorgu->execute([
        $data['tur'],
        $data['alici'],
        $data['baslik'],
        $data['icerik']
    ]);
    
    return $db->lastInsertId();
}

function updateBildirimLog($id, $success, $hata_mesaji = null) {
    global $db;
    
    $sorgu = $db->prepare("UPDATE bildirim_log SET 
        durum = ?, 
        hata_mesaji = ? 
        WHERE id = ?");
    
    $sorgu->execute([
        $success ? 'success' : 'failed',
        $hata_mesaji,
        $id
    ]);
}
?>