<?php
require_once '../config/db.php';

// UUID oluşturma fonksiyonu
function generateUUID() {
    return sprintf('%04x%04x-%04x-%04x-%04x-%04x%04x%04x',
        mt_rand(0, 0xffff), mt_rand(0, 0xffff),
        mt_rand(0, 0xffff),
        mt_rand(0, 0x0fff) | 0x4000,
        mt_rand(0, 0x3fff) | 0x8000,
        mt_rand(0, 0xffff), mt_rand(0, 0xffff), mt_rand(0, 0xffff)
    );
}

// Log fonksiyonu
function logYaz($mesaj) {
    $log_dosyasi = __DIR__ . '/fatura_log.txt';
    $tarih = date('Y-m-d H:i:s');
    file_put_contents($log_dosyasi, "[$tarih] $mesaj\n", FILE_APPEND);
}

try {
    $db->beginTransaction();
    
    // Vadesi gelen hosting hesaplarını bul
    $hosting_sorgu = $db->prepare("
        SELECT h.*, m.ad, m.soyad, m.eposta, hp.fiyat as paket_fiyat
        FROM hosting_hesaplari h
        LEFT JOIN musteriler m ON h.musteri_id = m.id
        LEFT JOIN hosting_paketleri hp ON h.paket_id = hp.id
        WHERE h.durum = 'Aktif' 
        AND h.sonraki_odeme <= CURDATE()
        AND h.fatura_donemi != 'Ücretsiz'
        AND NOT EXISTS (
            SELECT 1 FROM faturalar f 
            JOIN fatura_kalemleri fk ON f.id = fk.fatura_id 
            WHERE fk.hizmet_id = h.id 
            AND fk.hizmet_turu = 'Hosting'
            AND f.fatura_tarihi = CURDATE()
        )
    ");
    $hosting_sorgu->execute();
    $hosting_hesaplari = $hosting_sorgu->fetchAll(PDO::FETCH_ASSOC);
    
    // Vadesi gelen domainleri bul
    $domain_sorgu = $db->prepare("
        SELECT d.*, m.ad, m.soyad, m.eposta, df.yenileme_fiyat
        FROM domain_kayitlar d
        LEFT JOIN musteriler m ON d.musteri_id = m.id
        LEFT JOIN domain_fiyatlar df ON df.uzanti = SUBSTRING_INDEX(d.domain, '.', -2)
        WHERE d.durum = 'Aktif' 
        AND d.bitis_tarihi <= DATE_ADD(CURDATE(), INTERVAL 30 DAY)
        AND NOT EXISTS (
            SELECT 1 FROM faturalar f 
            JOIN fatura_kalemleri fk ON f.id = fk.fatura_id 
            WHERE fk.hizmet_id = d.id 
            AND fk.hizmet_turu = 'Domain'
            AND f.fatura_tarihi = CURDATE()
        )
    ");
    $domain_sorgu->execute();
    $domainler = $domain_sorgu->fetchAll(PDO::FETCH_ASSOC);
    
    // Müşteri bazında fatura oluştur
    $musteriler = [];
    
    // Hosting hesaplarını grupla
    foreach ($hosting_hesaplari as $hosting) {
        if (!isset($musteriler[$hosting['musteri_id']])) {
            $musteriler[$hosting['musteri_id']] = [
                'ad' => $hosting['ad'],
                'soyad' => $hosting['soyad'],
                'eposta' => $hosting['eposta'],
                'hosting' => [],
                'domain' => []
            ];
        }
        $musteriler[$hosting['musteri_id']]['hosting'][] = $hosting;
    }
    
    // Domainleri grupla
    foreach ($domainler as $domain) {
        if (!isset($musteriler[$domain['musteri_id']])) {
            $musteriler[$domain['musteri_id']] = [
                'ad' => $domain['ad'],
                'soyad' => $domain['soyad'],
                'eposta' => $domain['eposta'],
                'hosting' => [],
                'domain' => []
            ];
        }
        $musteriler[$domain['musteri_id']]['domain'][] = $domain;
    }
    
    // Her müşteri için fatura oluştur
    foreach ($musteriler as $musteri_id => $musteri) {
        if (empty($musteri['hosting']) && empty($musteri['domain'])) {
            continue;
        }
        
        // Fatura numarası oluştur
        $yil = date('Y');
        $sayac_sorgu = $db->prepare("INSERT INTO fatura_sayac (yil, son_numara) 
                                    VALUES (?, 0) 
                                    ON DUPLICATE KEY UPDATE son_numara = son_numara + 1");
        $sayac_sorgu->execute([$yil]);
        
        $sorgu = $db->prepare("SELECT son_numara FROM fatura_sayac WHERE yil = ?");
        $sorgu->execute([$yil]);
        $son_numara = $sorgu->fetchColumn();
        
        $fatura_no = sprintf("%s-%04d", $yil, $son_numara);
        
        // Fatura başlığını oluştur
        $fatura_id = generateUUID();
        $fatura_tarihi = date('Y-m-d');
        $son_odeme_tarihi = date('Y-m-d', strtotime('+7 days'));
        
        $sorgu = $db->prepare("INSERT INTO faturalar 
            (id, musteri_id, fatura_no, fatura_tarihi, son_odeme_tarihi) 
            VALUES (?, ?, ?, ?, ?)");
        
        $sorgu->execute([
            $fatura_id,
            $musteri_id,
            $fatura_no,
            $fatura_tarihi,
            $son_odeme_tarihi
        ]);
        
        $ara_toplam = 0;
        
        // Hosting kalemlerini ekle
        foreach ($musteri['hosting'] as $hosting) {
            $kalem_id = generateUUID();
            
            $sorgu = $db->prepare("INSERT INTO fatura_kalemleri 
                (id, fatura_id, hizmet_id, hizmet_turu, aciklama, birim, miktar, birim_fiyat, toplam) 
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)");
            
            // Paket fiyatını veya varsayılan fiyatı kullan
            $fiyat = $hosting['paket_fiyat'] ?: $hosting['fiyat'];
            $toplam = $fiyat;
            
            $sorgu->execute([
                $kalem_id,
                $fatura_id,
                $hosting['id'],
                'Hosting',
                $hosting['domain'] . ' Hosting Yenileme (' . $hosting['paket'] . ')',
                'Yıl',
                1,
                $fiyat,
                $toplam
            ]);
            
            $ara_toplam += $toplam;
            
            // Sonraki ödeme tarihini güncelle
            $yeni_tarih = date('Y-m-d', strtotime('+1 year', strtotime($hosting['sonraki_odeme'])));
            $sorgu = $db->prepare("UPDATE hosting_hesaplari SET sonraki_odeme = ? WHERE id = ?");
            $sorgu->execute([$yeni_tarih, $hosting['id']]);
        }
        
        // Domain kalemlerini ekle
        foreach ($musteri['domain'] as $domain) {
            $kalem_id = generateUUID();
            
            $sorgu = $db->prepare("INSERT INTO fatura_kalemleri 
                (id, fatura_id, hizmet_id, hizmet_turu, aciklama, birim, miktar, birim_fiyat, toplam) 
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)");
            
            // Domain uzantısına göre fiyatı al veya varsayılan fiyatı kullan
            $fiyat = $domain['yenileme_fiyat'] ?: 150.00;
            $toplam = $fiyat;
            
            $sorgu->execute([
                $kalem_id,
                $fatura_id,
                $domain['id'],
                'Domain',
                $domain['domain'] . ' Domain Yenileme',
                'Yıl',
                1,
                $fiyat,
                $toplam
            ]);
            
            $ara_toplam += $toplam;
        }
        
        // KDV hesapla ve faturayı güncelle
        $kdv_oran = 20.00;
        $kdv_tutar = $ara_toplam * ($kdv_oran / 100);
        $genel_toplam = $ara_toplam + $kdv_tutar;
        
        $sorgu = $db->prepare("UPDATE faturalar SET 
            ara_toplam = ?, 
            kdv_oran = ?, 
            kdv_tutar = ?, 
            genel_toplam = ? 
            WHERE id = ?");
        
        $sorgu->execute([
            $ara_toplam,
            $kdv_oran,
            $kdv_tutar,
            $genel_toplam,
            $fatura_id
        ]);
        
        // Online ödeme kaydı oluştur
        $online_odeme_id = generateUUID();
        $sorgu = $db->prepare("INSERT INTO online_odemeler (id, fatura_id, tutar) VALUES (?, ?, ?)");
        $sorgu->execute([$online_odeme_id, $fatura_id, $genel_toplam]);
        
        // Faturayı gönder
        require_once '../inc/phpmailer/PHPMailer.php';
        require_once '../inc/phpmailer/SMTP.php';
        require_once '../inc/phpmailer/Exception.php';
        
        $mail = new PHPMailer\PHPMailer\PHPMailer(true);
        
        try {
            // Server ayarları
            $mail->isSMTP();
            $mail->Host = 'mail.torossogutma.com.tr';
            $mail->SMTPAuth = true;
            $mail->Username = 'cem@torossogutma.com.tr';
            $mail->Password = 'Ts147258-+';
            $mail->SMTPSecure = PHPMailer\PHPMailer\PHPMailer::ENCRYPTION_STARTTLS;
            $mail->Port = 587;
            $mail->CharSet = 'UTF-8';
            
            // Gönderici ve alıcı
            $mail->setFrom('cem@torossogutma.com.tr', 'Epifiz Web Ajans');
            $mail->addAddress($musteri['eposta']);
            
            // İçerik
            $mail->isHTML(true);
            $mail->Subject = "Fatura #" . $fatura_no;
            
            // Online ödeme linki
            $online_odeme_link = SITE_URL . "/odeme.php?id=" . $online_odeme_id;
            
            $mail->Body = "
                <h2>Sayın {$musteri['ad']} {$musteri['soyad']},</h2>
                <p>#{$fatura_no} numaralı faturanız oluşturulmuştur.</p>
                <p><strong>Fatura Tarihi:</strong> " . date('d.m.Y', strtotime($fatura_tarihi)) . "</p>
                <p><strong>Son Ödeme Tarihi:</strong> " . date('d.m.Y', strtotime($son_odeme_tarihi)) . "</p>
                <p><strong>Tutar:</strong> " . number_format($genel_toplam, 2, ',', '.') . " ₺</p>
                <hr>
                <p>Banka hesap bilgilerimiz:</p>
                <p>
                    Banka: XXX Bank<br>
                    Şube: XXX Şubesi<br>
                    IBAN: TRXX XXXX XXXX XXXX XXXX XXXX XX
                </p>
                <div style='margin: 30px 0; text-align: center;'>
                    <a href='{$online_odeme_link}' 
                       style='background: #28a745; color: #fff; padding: 12px 25px; text-decoration: none; border-radius: 4px;'>
                        Kredi Kartı ile Öde
                    </a>
                </div>
                <p>Saygılarımızla,<br>Şirket Adı</p>
            ";
            
            $mail->send();
            logYaz("Fatura #{$fatura_no} oluşturuldu ve gönderildi: {$musteri['eposta']}");
        } catch (Exception $e) {
            logYaz("Fatura #{$fatura_no} mail gönderme hatası: " . $mail->ErrorInfo);
        }
    }
    
    $db->commit();
    logYaz("İşlem başarıyla tamamlandı.");
    
} catch (Exception $e) {
    $db->rollBack();
    logYaz("HATA: " . $e->getMessage());
}